# 옵저버 패턴 (Observer Pattern)

**옵저버 패턴**은 객체의 상태 변화를 관찰하는 **관찰자들**, 즉 옵저버들의 목록을 객체에 등록하여 객체의 **상태 변화**가 있을때마다 메소드를 통해 객체가 직접 목록의 각 옵저버들에게 **통지(알림)** 하는 디자인 패턴이다. </br>
</br>
간단히 말하자면 **어떤 객체**의 상태가 변할 때, 그와 연관된 객체들에게 **알림**을 보내는 디자인 패턴이다. </br>

> 다르게 표현하자면 여러 이벤트들을 감시하여, 이벤트가 발생할 때마다 미리 정의해둔 어떤 동작을 수행하게 해주는 패턴이다. </br>

## 구성

- `Observer` : Subject에서 변화했다고 알리기 위해 필요한 인터페이스
- `Subject` : Observer를 알고있는 주체. 옵저버들을 등록하고 제거하는데 필요한 인터페이스
- `ConcreateSubject` : 옵저버들에게 알려줘야할 상태를 저장, `notify` 함수 정의.
- `ConcreateObserver` : 알림을 받았을 때 수행하는 동작 정의

<img src = "https://user-images.githubusercontent.com/102718303/210763057-ad93cbbb-ef95-4d6d-bc4f-8fee10541c1b.jpg">



## 예시
코치와 크루가 있다고 하자. 크루들은 코치가 하는 일들을 모두 **알림(notify)** 받아야 한다. </br>
- 크루를 등록하고 해제, 알림을 보내는 **코치 인터페이스**와 자신의 상태를 업데이트하는 **크루 인터페이스**가 필요하다.

```java
public interface Coach {
  void register(Crew crew);
  void unregister(Crew crew);
  void nofifyCrew(String msg);
}

public interface Crew {
  void update(String msg);
}
```

- 코치인 연아 클래스를 구현한다.
  - 크루 리스트와 세가지 기능을 가진다.
  - notifyCrew 메소드를 각 기능에서 호출하는게 포인트!

```java
public class Yeonah implements Coach {
  private List<Crew> crews = new ArrayList<>();
  
  public void eatFood() {
    System.out.println("연아 코치가 밥을 먹었다.");
    notifyCrew("나 밥 먹었다.");
  }
  
   public void practice() {
    System.out.println("연아 코치가 연습을 했다.");
    notifyCrew("나 연습했다.");
  }
  
   public void upgradeCutie() {
    System.out.println("연아 코치가 귀여움을 강화했다.");
    notifyCrew("나 더 귀여워졌다.");
  }
  
  @Override
  public void register(Crew crew) {
    crews.add(crew);
  }
  
  @Override
  public void unregister(Crew crew) {
    crews.remove(crew);
  }
  
  @Override
  public void notifyCrew(Crew crew) {
    crews.forEach(crew -> crew.update(msg));
  }
}
```

- 크루인 티버 클래스를 구현한다. (다른 크루들도 구현하지만 생략) </br>

```java
public class Tiber implements Crew {
  @Override 
  public void update(String msg) {
    System.out.println("Tiber 수신 : " + msg);
  }
}
```

- 메인함수를 구현하고 실제 동작해보면,, </br>

```java
public class main {
  public static void main(String[] args) {
    Yeonah yeonah = new Teana();
    Crew tiber = new Tiber();
    Crew lewin = new Lewin();
    Crew jay = new Jay();
    
    yeonah.register(tiber);
    yeonah.register(lewin);
    yeonah.register(jay);
    
    yeonah.upgradeCutie(); // 등록한 3명의 크루 객체들에게 메세지(알림)가 전달된다.
  }
}
```

`Crew` 인터페이스가 `Observer` 인터페이스가 되고, `Coach` 인터페이스는 `Observalble` 인터페이스가 된다. </br>

    
## 장점
- 다른 객체의 상태 변화를 별도의 함수 호출 없이 즉각적으로 알 수 있다.
- 느슨한 결합으로 시스템이 유연하고 객체간의 의존성을 제거할 수 있다. </br>

> 느슨한 결합이란? </br>
> 서로에 대해 잘 모르는 것을 의미한다. (서로간의 공통규약들을 인터페이스에 정의하는 것처럼,,) </br>


## 단점
- 너무 많은 정보를 사용하게 되면, 상태 관리가 어려울 수 있다. </br>


    
## 실제 사용

- 자바에서 `Observer`인터페이스와 `Obsevable`클래스를 제공한다. (Obsevable은 클래스이다,,)
- `Obsevable` 클래스에 존재하는 notifyObserver 메소드를 통해 객체들에게 알림을 보낸다.
    
    
----
## Referance
- https://pjh3749.tistory.com/266
- https://kscory.com/dev/design-pattern/observer












